% Function: This is the program that generates an excel csv output with the 
% p-values and fold changes for all of the consistently significant 
% IP/Probe combinations 

% Inputs:
% ExpHitCell - a complicated cell array that contains information about the
%    identity, number, matrix index of the significant hits
% RepHitCell - a complicated cell array that contains information about the
%    identity, number, matrix index of the consistently bad replicates 
% PValueCell - a cell array, where each entry is another cell array that 
%    contains the matrices with the significance values for a each 
%    comparison in a given experiment
% Log2FCCell - a cell array, where each entry is another cell array that 
%    contains the matrices with the log2fold change values for a each 
%    comparison in a given experiment
% ExpNum - a numerical value for the number of experiments performed
% ConditionNum - a numerical value for the number of conditions/comparisons
%    performed
% IPProbeNum - a numerical value with the number of IP/Probe combinations
%    measured
% ConditionId - a cell array where the entries are strings naming the
%    different conditions/combinations being performed
% Filename - a string naming the file of the output

% Outputs:
% The only output is a formatted excel csv file that lists the consistently
% significant hits (for a few consistency levels), along with the p-values
% and fold changes for each of these significant IP/Probe hits

function [] = ExcelHitOutput(ExpHitCell,RepHitCell,PValueCell,...
    Log2FCCell,ExpNum,ConditionNum,IPProbeNum,ConditionId,FileName)

fid = fopen(FileName,'wt');
StringFormat = strcat('%s,',repmat('%d,',1,2*ExpNum),'%d,%s\n');

for alpha = 1:ConditionNum
    
    fprintf(fid,'%s\n',ConditionId{alpha});
    fprintf(fid,'%s\n',strcat(num2str(ExpNum),'of',num2str(ExpNum)));
    
    if ExpNum <= 2
        NumUnionsPrint = 1;
    elseif ExpNum <= 5
        NumUnionsPrint = 2;
    else
        NumUnionsPrint = 3;
    end
  
 
    for beta = 1:NumUnionsPrint
    
        Condition_a = alpha;
        CombNum = 1;
        NumUnions = beta;

        [HitsId,HitsIndex,~] = SimpleHits(ExpHitCell,...
            Condition_a,ExpNum,CombNum,NumUnions);
        % Function - ExcelOutput

% Commented out code. This code can be used to troubleshoot and identify 
% IP/Probes with consistent variation between replicates.        
%         [~,BadIndex] = SimpleHitsBadPPI(RepHitCell,ExpNum,CombNum,...
%             NumUnions);
%         % Function - ExcelOutput
%         
%         [~,HitBad_Intersect] = intersect(HitsIndex,BadIndex);

        PList = zeros(ExpNum,IPProbeNum);
        FCList = zeros(ExpNum,IPProbeNum);

        for i = 1:ExpNum
            PList(i,:) = PValueCell{i}{Condition_a}(:);
            FCList(i,:) = Log2FCCell{i}{Condition_a}(:);
        end

        HitsPValues = PList(:,HitsIndex)';
        HitsFCValues = 2.^FCList(:,HitsIndex)';

        Data_a = cell(length(HitsIndex),3+2*ExpNum);

        [Data_a(:,1)] = HitsId;
        [Data_a(:,2:1+ExpNum)] = num2cell(HitsPValues);
        [Data_a(:,2+ExpNum:1+2*ExpNum)] = num2cell(HitsFCValues);
        [Data_a(:,2+2*ExpNum)] = num2cell(NumUnions);
        
% Commented out code. This code can be used to troubleshoot and identify 
% IP/Probes with consistent variation between replicates.         
%         [Data_a(HitBad_Intersect,3+2*ExpNum)] = {'BadRepHit'};

        for j=1:size(Data_a,1)
            fprintf(fid,StringFormat, Data_a{j,:});
        end
        
        if beta == 1 && NumUnionsPrint >1
            fprintf(fid,'%s\n',strcat(num2str(ExpNum-1),'of',...
                num2str(ExpNum),'Union',num2str(ExpNum),'of',...
                num2str(ExpNum)));
            
        end
        
        if beta == 2 && NumUnionsPrint >2
            fprintf(fid,'%s\n',strcat(num2str(ExpNum-2),'U',...
                num2str(ExpNum-1),'U',num2str(ExpNum)));
            
        end
    
    end
end
fclose(fid);